<?php

declare(strict_types=1);

namespace Gls\GlsPoland\Consignment\DTO;

use Gls\GlsPoland\Carrier\CarrierType;
use Gls\GlsPoland\Carrier\TimeDefiniteService;
use Symfony\Component\Validator\Constraints as Assert;

class ServiceDetails
{
    /**
     * @var string|null
     *
     * @Assert\Length(max="20")
     */
    private $parcelShopId;

    /**
     * @var float|null
     *
     * @Assert\PositiveOrZero()
     */
    private $codAmount;

    /**
     * @var bool|null
     */
    private $documentReturn = false;

    /**
     * @var bool|null
     */
    private $saturdayDelivery = false;

    /**
     * @var TimeDefiniteService|null
     */
    private $timeDefiniteService;

    /**
     * @var bool|null
     */
    private $shopReturn = false;

    public function getParcelShopId(): ?string
    {
        return $this->parcelShopId;
    }

    public function setParcelShopId(?string $parcelShopId): self
    {
        $this->parcelShopId = $parcelShopId;

        return $this;
    }

    public function isShopDelivery(): bool
    {
        return null !== $this->getParcelShopId();
    }

    public function isDocumentReturn(): bool
    {
        return true === $this->documentReturn;
    }

    public function setDocumentReturn(?bool $isDocumentReturn): self
    {
        $this->documentReturn = $isDocumentReturn;

        return $this;
    }

    public function getCashOnDeliveryAmount(): ?float
    {
        return $this->codAmount;
    }

    public function setCashOnDeliveryAmount(?float $amount): self
    {
        $this->codAmount = $amount;

        return $this;
    }

    public function isCashOnDelivery(): bool
    {
        return 0. < (float) $this->getCashOnDeliveryAmount();
    }

    public function isSaturdayDelivery(): bool
    {
        return true === $this->saturdayDelivery;
    }

    public function setSaturdayDelivery(?bool $isSaturdayDelivery): self
    {
        $this->saturdayDelivery = $isSaturdayDelivery;

        return $this;
    }

    public function getTimeDefiniteService(): ?TimeDefiniteService
    {
        return $this->timeDefiniteService;
    }

    public function setTimeDefiniteService(?TimeDefiniteService $service): self
    {
        $this->timeDefiniteService = $service;

        return $this;
    }

    public function isDeliveryBefore10(): bool
    {
        return TimeDefiniteService::DELIVERY_BEFORE_10 === (string) $this->timeDefiniteService;
    }

    public function isDeliveryBefore12(): bool
    {
        return TimeDefiniteService::DELIVERY_BEFORE_12 === (string) $this->timeDefiniteService;
    }

    public function isShopReturn(): bool
    {
        return true === $this->shopReturn;
    }

    public function setShopReturn(?bool $isShopReturn): self
    {
        $this->shopReturn = $isShopReturn;

        return $this;
    }

    public function isExpressDelivery(): bool
    {
        return $this->isSaturdayDelivery()
            || $this->isDeliveryBefore10()
            || $this->isDeliveryBefore12();
    }

    public function getCarrierType(): CarrierType
    {
        return $this->isShopDelivery()
            ? CarrierType::from(CarrierType::SHOP_DELIVERY)
            : CarrierType::from(CarrierType::COURIER);
    }
}
